# -*- coding: utf-8 -*-

# multicolumn.py

__author__ = 'jacques.tisseau@enib.fr'

from docutils import nodes
from docutils.parsers.rst import directives
from docutils.parsers.rst.directives import tables;

from enibook import EniBook
from block import collapseblock

#------------------------------------------------------------------------------
# Multicolumn
#------------------------------------------------------------------------------
MULTICOLUMN_HTML_BEGIN = u"""
<!-- begin %(id)s -->
"""

MULTICOLUMN_HTML_BTN_BEGIN = u"""
<span id="%(id)s-multicolumn-head" name="%(name)s-multicolumn-head">
    <span id="%(id)s-multicolumn-button" name="%(name)s-multicolumn-button"
        class="btn btn-warning glyphicons glyphicons-menu-hamburger rotate90" 
        style="font-size: 150%%;"></span>
    <span id="%(id)s-multicolumn-nbcols" name="%(name)s-multicolumn-nbcols"
        class="multicolumn-nbcols %(class)s-multicolumn-nbcols">%(multicol)s</span>
    <span class="multicolumn-title %(class)s-multicolumn-title"
        id="%(id)s-multicolumn-title" name="%(name)s-multicolumn-title">%(title)s</span>
    <span id="%(id)s-multicolumn-btns" name="%(name)s-multicolumn-btns"
        data-role="controlgroup" data-type="horizontal">
"""

MULTICOLUMN_HTML_BTN = u"""
    <button class="ui-btn ui-mini" numero="%(numero)s"
        onclick="EniBook['%(id)s'].btnActive(this); EniBook['%(id)s'].feedback(this);">%(numero)s</button>
"""

MULTICOLUMN_HTML_BTN_END = u"""
    </span>
</span> 
<div id="%(id)s-multicolumn" name="%(name)s-multicolumn" 
    class="ui-grid-%(columns)s ui-responsive multicolumn %(class)s-multicolumn"
    style="margin-top: 0px;"
    multicol="%(multicol)s" collapse="%(collapse)s" directive="%(directive)s">
"""

MULTICOLUMN_HTML_END = u"""
</div>
<div class="multicolumn-footer %(class)s-multicolumn-footer" 
    id="%(id)s-multicolumn-footer" name="%(name)s-multicolumn-footer">
    <span class="btn btn-warning glyphicons glyphicons-chevron-up"
        onclick="$.mobile.silentScroll($('#%(id)s-multicolumn-btns').offset().top);"></span>
</div>
<!-- end %(id)s -->
<script type="text/javascript"> 
    $("#%(id)s-multicolumn-button").on("click", function(event) { event.stopPropagation(); EniBook['%(id)s'].update(); } );
    //$(document).ready( function() { EniBook.init.push( function() { EniBook['%(id)s'] = new Multicolumn("%(id)s"); }) });
    EniBook.init.push( function() { EniBook['%(id)s'] = new Multicolumn("%(id)s"); } );
</script>
"""
#------------------------------------------------------------------------------

#-----------------------------------------------------
class multicolumn(nodes.General, nodes.Element):
    def __init__(self,options):
        super(multicolumn,self).__init__()
        self.options = options
#-----------------------------------------------------

#-----------------------------------------------------
def visit_html_multicolumn(self, node):
    res = node.html_begin % node.options
    self.body.append(res)

def depart_html_multicolumn(self,node):
    res = node.html_end % node.options
    self.body.append(res)
#-----------------------------------------------------

#-----------------------------------------------------
class Multicolumn(EniBook):
    required_arguments = 1
    optional_arguments = 0
    final_argument_whitespace = True
    has_content = True
    option_spec = {
        'class'    : directives.class_option,
        'name'     : directives.unchanged,
        'title'    : directives.unchanged,
        'collapse' : collapseblock
    }

    columns = {"1" : "solo", "2" : "a", "3" : "b", "4" : "c", "5" : "d"}
	
    def run(self):
        super(Multicolumn,self).run()
        env = self.state.document.settings.env
        
        nbcols = nbcolumn(self.arguments[0])
        self.options['columns']  = self.columns[nbcols]
        self.options['multicol'] = nbcols
        env.colnumber = -1
        env.multicol = int(nbcolumn(self.arguments[0]))
        
        if 'collapse' not in self.options :
            self.options['collapse'] = 'true'
        if 'title' not in self.options :
            self.options['title'] = ''

        node = multicolumn(self.options)
        node.html_begin = ''
        if env.multicol > 1 :
            node.html_begin += MULTICOLUMN_HTML_BTN_BEGIN
            for i in range(env.multicol) :
                node.html_begin += MULTICOLUMN_HTML_BTN % {'id' : self.options['id'], 'numero' : i+1}
            node.html_begin += MULTICOLUMN_HTML_BTN_END
        node.html_begin += MULTICOLUMN_HTML_BEGIN
        node.html_end    = MULTICOLUMN_HTML_END
        self.state.nested_parse(self.content, self.content_offset, node)

        env.colnumber = -1
        
        return [node]

def nbcolumn(argument):
    return directives.choice(argument, ("1","2","3","4","5"))
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# Column
#------------------------------------------------------------------------------
COLUMN_HTML_BEGIN = u"""
<div id="%(id)s-column-%(numero)s" name="%(name)s" 
    class="ui-block-%(numero)s column %(class)s-column" 
    collapse="%(collapse)s" directive="%(directive)s">
    <h4>%(number)s. %(title)s</h4>
"""

COLUMN_HTML_END = u"""</div>
"""
#------------------------------------------------------------------------------

#-----------------------------------------------------
class column(nodes.General, nodes.Element):
    def __init__(self,options):
        super(column,self).__init__()
        self.options = options
#-----------------------------------------------------

#-----------------------------------------------------
def visit_html_column(self, node):
    if not isinstance(node.parent,multicolumn) :
        subdirective  = 'Column'
        directive     = 'Multicolumn'
        eniBookError(subdirective,directive)
    res = node.html_begin % node.options
    self.body.append(res)

def depart_html_column(self,node):
    res = node.html_end % node.options
    self.body.append(res)
#-----------------------------------------------------

#-----------------------------------------------------
class Column(EniBook):
    required_arguments = 0
    optional_arguments = 0
    final_argument_whitespace = True
    has_content = True
    option_spec = {
        'class'    : directives.class_option,
        'name'     : directives.unchanged,
        'collapse' : collapseblock,
        'title'    : directives.unchanged
    }

    def run(self):
        super(Column,self).run()
        numero, number = self.getNumero()
        self.options['numero'] = numero
        self.options['number'] = number + 1
        
        if 'collapse' not in self.options :
            self.options['collapse'] = 'true'

        if 'title' not in self.options :
            self.options['title'] = ""

        node = column(self.options)
        node.html_begin = COLUMN_HTML_BEGIN
        node.html_end   = COLUMN_HTML_END
        self.state.nested_parse(self.content, self.content_offset, node)
        return [node]

    def getNumero(self) :
        env = self.state.document.settings.env
        env.colnumber += 1
        numero = chr(ord("a") + env.colnumber%env.multicol)
        return numero, env.colnumber
#------------------------------------------------------------------------------

#------------------------------------------------------------------------------
# WebCSVTable
#------------------------------------------------------------------------------
class webcsvtable(nodes.table,nodes.Element) :
    def __init__(self,options):
        super(webcsvtable,self).__init__()
        self.options = options

def visit_html_webcsvtable(self, node):
    self.context.append(self.compact_p)
    self.compact_p = True
    classes = ' '.join(['docutils', self.settings.table_style]).strip()
    res = self.starttag(node, 'table', CLASS=classes)
    data = ' data-role="table" data-mode="columntoggle" '
    data += '''data-column-btn-text="<span class='btn btn-warning glyphicons glyphicons-table'></span>"'''
    res = res.replace('>',data+'>')
    self.body.append(res)

def depart_html_webcsvtable(self, node):
    self.compact_p = self.context.pop()
    self.body.append('</table>\n')
    self.body = jqmTable(self.body)
                
def jqmTable(body) :        
    res = '$$$$'.join(body)
    index = res.rfind('<thead')
    tail = ''
    if (index != -1) :
        tail = res[index:]
        res = res[:index]
        priority = 1
        index = 0
        while index != -1 :
            index = tail.find('<th ',index)
            if index != -1 :
                begin = tail[:index]
                end   = tail[index+4:]
                tail = begin + '<th data-priority="' + str(priority) + '" ' + end
                if priority < 6 : priority += 1
                index = index + len('<th data-priority="1" ')
    res = res + tail
    return res.split('$$$$')

def visit_latex_webcsvtable(self, node):
    pass

def depart_latex_webcsvtable(self, node):
    pass

class WebCSVTable(tables.CSVTable) :
    def run(self) :
        nodelist = super(WebCSVTable,self).run()
        node = nodelist[0]

        nodeweb = webcsvtable()
        nodeweb['classes'].append('ui-responsive ui-shadow webcsvtable')
        for i in range(len(node)) :
            nodeweb.append(node[i])
        return [nodeweb]
