# -*- coding: utf-8 -*-

# diagram.py

__author__ = 'jacques.tisseau@enib.fr'

from docutils import nodes
from docutils.parsers.rst import directives, roles

from enibook import EniBook

#------------------------------------------------------------------------------
# Graph
#------------------------------------------------------------------------------
GRAPH_HTML_BEGIN_1 = u"""
<span class="btn btn-warning glyphicons glyphicons-flowchart"
    onclick="EniBook['%(id)s'].toggle();"></span>
<span id="%(id)s-graph-graphname" name="%(name)s-graph-graphname"
    class="graph-graphname %(class)s-graph-graphname">%(graph)s : </span>
<span id="%(id)s-graph-filename" name="%(name)s-graph-filename" 
    class="graph-filename %(class)s-graph-filename">%(file)s</span>
<div id="%(id)s-graph" name="%(name)s-graph" tocode="%(tocode)s"
    class="graph %(class)s-graph" directive="%(directive)s" type="%(type)s" 
    modeEnibook="%(modeEnibook)s" typeEnibook = "%(typeEnibook)s" title="%(title)s"
    file="%(file)s" fileref="%(fileref)s" version="%(version)s" collapse="%(collapse)s">
    <table width=100%% data-role="table" class="ui-responsive">
        <thead>
            <th width=15%%></th>
            <th width=85%%></th>
        </thead>
        <tbody>
            <tr>
                <td>
                    <div id="%(id)s-graph-palette" name="%(name)s-graph-palette"
                        class="graph-palette %(class)s-graph-palette" 
                        style="height:500px;">
"""

QUERYTREE_HTML = u"""
                        <h4>Tables</h4>
                        <div>
                            <div id="%(id)s-graph-palette-0" name="%(name)s-graph-palette-0"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Opérateurs unaires</h4>
                        <div>
                            <div id="%(id)s-graph-palette-1" name="%(name)s-graph-palette-1"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Opérateurs binaires</h4>
                        <div>
                            <div id="%(id)s-graph-palette-2" name="%(name)s-graph-palette-2"
                                style="min-height: 290px;"></div>
                        </div>
"""

STATECHART_HTML = u"""
                        <h4>Nœuds</h4>
                        <div>
                            <div id="%(id)s-graph-palette-0" name="%(name)s-graph-palette-0"
                                style="min-height: 290px;"></div>
                        </div>
"""

DATATREE_HTML = u"""
                        <h4>Nœuds</h4>
                        <div>
                            <div id="%(id)s-graph-palette-0" name="%(name)s-graph-palette-0"
                                style="min-height: 290px;"></div>
                        </div>
"""

MEMORY_HTML = u"""
                        <h4>Zones</h4>
                        <div>
                            <div id="%(id)s-graph-palette-0" name="%(name)s-graph-palette-0"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Contextes</h4>
                        <div>
                            <div id="%(id)s-graph-palette-1" name="%(name)s-graph-palette-1"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Objets simples</h4>
                        <div>
                            <div id="%(id)s-graph-palette-2" name="%(name)s-graph-palette-2"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Objets composés</h4>
                        <div>
                            <div id="%(id)s-graph-palette-3" name="%(name)s-graph-palette-3"
                                style="min-height: 290px;"></div>
                        </div>
"""

LOGIC_HTML = u"""
                        <h4>Entrées/Sorties</h4>
                        <div>
                            <div id="%(id)s-graph-palette-0" name="%(name)s-graph-palette-0"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Portes 1/1</h4>
                        <div>
                            <div id="%(id)s-graph-palette-1" name="%(name)s-graph-palette-1"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Portes 2/1</h4>
                        <div>
                            <div id="%(id)s-graph-palette-2" name="%(name)s-graph-palette-2"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Portes 3/1</h4>
                        <div>
                            <div id="%(id)s-graph-palette-3" name="%(name)s-graph-palette-3"
                                style="min-height: 290px;"></div>
                        </div>
"""

FLOWCHART_HTML = u"""
                        <h4>Entrées/Sorties</h4>
                        <div>
                            <div id="%(id)s-graph-palette-0" name="%(name)s-graph-palette-0"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Instructions</h4>
                        <div>
                            <div id="%(id)s-graph-palette-1" name="%(name)s-graph-palette-1"
                                style="min-height: 290px;"></div>
                        </div>
                        <!-- <h4>Fonctions</h4>
                        <div>
                            <div id="%(id)s-graph-palette-2" name="%(name)s-graph-palette-2"
                                style="min-height: 290px;"></div>
                        </div> -->
                        <h4>Notes</h4>
                        <div>
                            <div id="%(id)s-graph-palette-2" name="%(name)s-graph-palette-2"
                                style="min-height: 290px;"></div>
                        </div>
"""

GRAPH_HTML_BEGIN_2 = u"""
                    </div>
                </td>
                <td>
                    <div id="%(id)s-graph-diagram" name="%(name)s-graph-diagram" 
                        class="graph-diagram %(class)s-graph-diagram"
                        style="height:500px; border:1px solid green;"></div>
                </td>
            </tr>
        </tbody>
    </table>
    <table width=100%% data-role="table" class="ui-responsive">
        <thead>
            <th width=15%%>Vue d'ensemble</td>
            <th width=85%%>
                <span id="%(id)s-file-btns"
                      data-inline="true" data-role="controlgroup" data-type="horizontal" data-native-menu="false">
                    <button class='btn btn-info' id='%(id)s-file-btn-reset' style='background-color : rgb(144,209,223);'
                        title="initialiser le diagramme"
                        onclick="EniBook['%(id)s'].update();"><span class='glyphicons glyphicons-refresh'></span></button>
                    <button class='btn btn-info' id="%(id)s-file-btn-download"  style='background-color : rgb(144,209,223);'  
                        title="charger un diagramme au format JSON"
                        onclick="EniBook['%(id)s'].openJSONDiagram();"><span class='glyphicons glyphicons-disk-open'></span></button>
                    <button class='btn btn-info' id="%(id)s-file-btn-upload"  style='background-color : rgb(144,209,223);' 
                        title="sauvegarder le diagramme au format JSON"
                        onclick="EniBook['%(id)s'].saveJSONDiagram();"><span class='glyphicons glyphicons-disk-save'></span></button>
                    <button class='btn btn-info' id="%(id)s-file-btn-print"  style='background-color : rgb(144,209,223);' 
                        title="imprimer le diagramme"
                        onclick="EniBook['%(id)s'].printDiagram();"><span class='glyphicons glyphicons-print'></span></button>
                </span>
                <span id="%(id)s-run-btns"
                      data-inline="true" data-role="controlgroup" data-type="horizontal" data-native-menu="false">
                    <button class='btn btn-warning' id='%(id)s-file-btn-grid' style='background-color : rgb(159,159,202);'
                        title="grille"
                        onclick="EniBook['%(id)s'].toggleGrid();"><span class='glyphicons glyphicons-stats'></span></button>
                    <button class='btn btn-warning' id="%(id)s-file-btn-upload"  style='background-color : rgb(159,159,202);' 
                        title="sauvegarder le diagramme au format SVG"
                        onclick="EniBook['%(id)s'].saveSVGDiagram();"><span class='glyphicons glyphicons-disk-save'></span></button>
                    <button class='btn btn-warning' id="%(id)s-file-btn-upload"  style='background-color : rgb(159,159,202);' 
                        title="sauvegarder le diagramme au format PNG"
                        onclick="EniBook['%(id)s'].savePNGDiagram();"><span class='glyphicons glyphicons-screenshot'></span></button>
                    <button class='btn' id="%(id)s-diagram-btn-help"
                        style="background-color: rgb(159,159,202);"
                        title="aide"
                        onclick="EniBook['%(id)s'].help();"><span class='glyphicons glyphicons-circle-question-mark'></span></button>
                </span>
                <span id="%(id)s-run-btns-1"
                      data-inline="true" data-role="controlgroup" data-type="horizontal" data-native-menu="false">
                    <button class='btn btn-success' id="%(id)s-file-btn-clear-output" style='background-color : rgb(162,169,63);'  
                        title="initialiser les sorties"
                        onclick="EniBook['%(id)s'].updateOutput();"><span class='glyphicons glyphicons-refresh'></span></button>
                    <button class='btn btn-success' id="%(id)s-file-btn-download" style='background-color : rgb(162,169,63);'  
                        title="analyser le diagramme"
                        onclick="EniBook['%(id)s'].feedback();"><span class='glyphicons glyphicons-play'></span></button>
                </span>
            </th>
        </thead>
        <tbody>
            <tr>
                <td valign="top">
                    <div id="%(id)s-graph-overview" name="%(name)s-graph-overview"
                        class="graph-overview %(class)s-graph-overview" 
                        style="height:150px; border:1px solid blue;">
                    </div>
                </td>
                <td valign="top">
                    <textarea id="%(id)s-graph-code" name="%(name)s-graph-code"
                        style="display:none;" placeholder="code JSON">%(code)s</textarea>
                    <textarea id="%(id)s-graph-fileref" name="%(name)s-graph-fileref"
                        style="display:none;" placeholder="code JSON">%(fileref)s</textarea>
                    <textarea id="%(id)s-graph-filecode" name="%(name)s-graph-filecode"
                        style="display:none;" placeholder="code">%(filecode)s</textarea>
                    <div id="%(id)s-graph-feedback" name="%(name)s-graph-feedback"
                        class="graph-feedback %(class)s-graph-feedback"></div>
                    <div id="%(id)s-graph-div" class="graph-div"></div>
                    <div id="%(id)s-graph-modal" class="graph-modal"></div>
                </td>
            </tr>
        </tbody>
    </table>
"""

GRAPH_HTML_END = u"""
</div>
<script type="text/javascript">
    //$(document).ready( function() { EniBook["%(id)s"] = new %(javascript-object)s("%(id)s"); } );
    //$(document).ready( function() { EniBook.init.push( function() { EniBook["%(id)s"] = new %(javascript-object)s("%(id)s"); }) });
    EniBook.init.push( function() { EniBook["%(id)s"] = new %(javascript-object)s("%(id)s"); } );
</script>
"""

DIAGRAM_PYTHON_CONFIG = u"""
pyEniBook['assessing']['%(directive)s']['%(id)s'] = {
    'title'     : u"%(title)s",
    'questions' : 1,
    'choices'   : 1,
    'items'     : [],
    'user'      : ""
}
"""

GRAPH_LATEX_BEGIN = u"""
"""

GRAPH_LATEX_END = u"""
"""

#------------------------------------------------------------------------------
class graph(nodes.General, nodes.Element):
    def __init__(self,options):
        super(graph,self).__init__()
        self.options = options

def visit_html_graph(self, node):
    res = node.html_begin % node.options
    self.body.append(res)

def depart_html_graph(self,node):
    res = node.html_end % node.options
    self.body.append(res)

def visit_latex_graph(self, node):
    pass
def depart_latex_graph(self,node):
    pass
#-----------------------------------------------------
def versiongraph(argument) :
    return directives.choice(argument, ("full","light"))

def typegraph(argument) :
    return directives.choice(argument, 
        ("analogic","datatree","flowchart","grafcet","logic","memory","petri","querytree","statechart","uml"))

graphName = {
    "analogic"  : u"Circuit analogique",
    "datatree"  : u"Arbre de données",
    "flowchart" : u"Flot d'instructions",
    "grafcet"   : u"Grafcet",
    "logic"     : u"Circuit logique",
    "memory"    : u"Mémoire",
    "petri"     : u"Réseau de Petri",
    "querytree" : u"Arbre de requêtes",
    "statechart": u"Diagramme d'états",
    "uml"       : u"Diagramme Uml"
}

objectJavascript = {
    "analogic"  : "Analogic",
    "datatree"  : "DataTree",
    "flowchart" : "FlowChart",
    "grafcet"   : "Grafcet",
    "logic"     : "Logic",
    "memory"    : "Memory",
    "petri"     : "Petri",
    "querytree" : "QueryTree",
    "statechart": "StateChart",
    "uml"       : "Uml"
}

class Graph(EniBook):
    required_arguments = 1
    optional_arguments = 0
    final_argument_whitespace = True
    has_content = True
    option_spec = {
        'class'   : directives.class_option,
        'name'    : directives.unchanged,
        'title'   : directives.unchanged,
        'file'    : directives.path,
        'fileref' : directives.path,
        'filecode': directives.path,
        'version' : versiongraph,
        'collapse': directives.flag
    }

    def run(self):
        super(Graph,self).run()
        
        if typegraph(self.arguments[0]) :
            self.options['type']  = self.arguments[0]
            self.options['graph'] = graphName[self.arguments[0]]
            self.options['javascript-object'] = objectJavascript[self.arguments[0]]

        if 'version' not in self.options :
            self.options['version'] = 'full'

        if 'collapse' not in self.options :
            self.options['collapse'] = 'false'
        else :
            self.options['collapse'] = 'true'

        if 'tocode' not in self.options :
            self.options['tocode'] = 'false'
        else :
            self.options['tocode'] = 'true'

        if 'file' not in self.options :
            self.options['file'] = self.options["id"] + ".json"
            self.options['code'] = ""
        else :
            if self.options['file'] != "" :
                text = self.include(self.options['file'])
                self.options['code'] = text
            else :
                self.options['code'] = ""

        import os.path
        self.options['file'] = os.path.basename(self.options['file'])

        if 'fileref' not in self.options :
            self.options['fileref'] = ""
        else :
            self.options['fileref'] = self.include(self.options['fileref'])
        
        if 'filecode' not in self.options :
            self.options['filecode'] = ""
            self.options['tocode'] = 'false'
        else :
            self.options['filecode'] = self.include(self.options['filecode'])
            self.options['tocode'] = 'true'
        
        node = graph(self.options)
        node.html_begin  = GRAPH_HTML_BEGIN_1
        if self.options['type'] == 'logic' : 
            node.html_begin += LOGIC_HTML
        elif self.options['type'] == 'flowchart' : 
            node.html_begin += FLOWCHART_HTML
        elif self.options['type'] == 'memory' : 
            node.html_begin += MEMORY_HTML
        elif self.options['type'] == 'datatree' : 
            node.html_begin += DATATREE_HTML
        elif self.options['type'] == "querytree" : 
            node.html_begin += QUERYTREE_HTML
        elif self.options['type'] == "statechart" : 
            node.html_begin += STATECHART_HTML
        else :
            pass

        node.html_begin += GRAPH_HTML_BEGIN_2
        node.html_end    = GRAPH_HTML_END
        node.latex_begin = GRAPH_LATEX_BEGIN
        node.latex_end   = GRAPH_LATEX_END
        self.state.nested_parse(self.content, self.content_offset, node)

        if self.options['modeEnibook'] != "learning" and os.path.exists("pyenibook.cfg") :
            file = open("pyenibook.cfg","a")
            txt = DIAGRAM_PYTHON_CONFIG % self.options
            txt = txt.encode("utf8")
            file.write(txt)
            file.close()

        return [node]
#------------------------------------------------------------------------------



#------------------------------------------------------------------------------
# Diagram
#------------------------------------------------------------------------------
DIAGRAM_HTML_BEGIN = u"""
<div id="%(id)s-diagram" name="%(name)s-diagram" 
    class="diagram %(class)s-diagram" directive="%(directive)s"
    type="%(type)s" file="%(file)s" version="%(version)s">
    <table width=100%% data-role="table" class="ui-responsive">
        <thead>
            <th width=15%%>Composants</th>
            <th width=85%%>
                <span class="btn btn-warning glyphicons glyphicons-flowchart"
                onclick="showPalette('%(id)s');"></span> Circuit logique : 
                <span id="%(id)s-diagram-file" name="%(name)s-diagram-file" 
                    class="diagram-file %(class)s-diagram-file">%(file)s</span></th>
        </thead>
        <tbody>
            <tr>
                <td>
                    <div id="%(id)s-diagram-palette" name="%(name)s-diagram-palette"
                        class="diagram-palette %(class)s-diagram-palette" 
                        style="height:500px;">
                        <h4>Entrées/Sorties</h4>
                        <div>
                            <div id="%(id)s-diagram-palette-0" name="%(name)s-diagram-palette-1"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Portes 1/1</h4>
                        <div>
                            <div id="%(id)s-diagram-palette-1" name="%(name)s-diagram-palette-2"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Portes 2/1</h4>
                        <div>
                            <div id="%(id)s-diagram-palette-2" name="%(name)s-diagram-palette-2"
                                style="min-height: 290px;"></div>
                        </div>
                        <h4>Portes 3/1</h4>
                        <div>
                            <div id="%(id)s-diagram-palette-3" name="%(name)s-diagram-palette-3"
                                style="min-height: 290px;"></div>
                        </div>
                    </div>
                </td>
                <td>
                    <div id="%(id)s-diagram-diagram" name="%(name)s-diagram-diagram"
                        class="diagram-diagram %(class)s-diagram-diagram" 
                        style="height:500px; border:1px solid green;"></div>
                </td>
            </tr>
        </tbody>
    </table>
    <table width=100%% data-role="table" class="ui-responsive">
        <thead>
            <th width=15%%>Vue d'ensemble</td>
            <th width=85%%>
                <span id="%(id)s-file-btns"
                      data-inline="true" data-role="controlgroup" data-type="horizontal" data-native-menu="false">
                    <button class='btn btn-info' id='%(id)s-file-btn-reset' style='background-color : rgb(144,209,223);'
                        title="initialiser le diagramme"
                        command="<kbd>Ctrl+N</kbd>"
                        icone="<span class='glyphicons glyphicons-refresh'></span>"
                        longtitle="initialiser l'éditeur"
                        onclick="newDiagram('%(id)s');"><span class='glyphicons glyphicons-refresh'></span></button>
                    <button class='btn btn-info' id="%(id)s-file-btn-download"  style='background-color : rgb(144,209,223);'  
                        title="charger un diagramme au format JSON"
                        command="<kbd>Ctrl+O</kbd>"
                        icone="<span class='glyphicons glyphicons-disk-open'></span>"
                        longtitle="copier le contenu d'un fichier dans l'éditeur"
                        onclick="openJSONDiagram('%(id)s');"><span class='glyphicons glyphicons-disk-open'></span></button>
                    <button class='btn btn-info' id="%(id)s-file-btn-upload"  style='background-color : rgb(144,209,223);' 
                        title="sauvegarder le diagramme au format JSON"
                        command="<kbd>Ctrl+S</kbd>"
                        icone="<span class='glyphicons glyphicons-disk-save'>"
                        longtitle="sauvegarder le contenu de l'éditeur dans un fichier"
                        onclick="saveJSONDiagram('%(id)s');"><span class='glyphicons glyphicons-disk-save'></span></button>
                    <button class='btn btn-info' id="%(id)s-file-btn-print"  style='background-color : rgb(144,209,223);' 
                        title="imprimer le diagramme"
                        command="<kbd>Ctrl+P</kbd>"
                        icone="<span class='glyphicons glyphicons-print'></span>"
                        longtitle="imprimer le contenu de l'éditeur"
                        onclick="printDiagram('%(id)s');"><span class='glyphicons glyphicons-print'></span></button>
                </span>
                <span id="%(id)s-run-btns"
                      data-inline="true" data-role="controlgroup" data-type="horizontal" data-native-menu="false">
                    <button class='btn btn-warning' id='%(id)s-file-btn-grid' style='background-color : rgb(234,138,0);'
                        title="grille"
                        command="<kbd>Ctrl+N</kbd>"
                        icone="<span class='glyphicons glyphicons-stats'></span>"
                        longtitle="grille"
                        onclick="gridToggle('%(id)s');"><span class='btn-warning glyphicons glyphicons-stats'></span></button>
                    <button class='btn btn-warning' id="%(id)s-file-btn-upload"  style='background-color : rgb(234,138,0);' 
                        title="sauvegarder le diagramme au format SVG"
                        command="<kbd>Ctrl+S</kbd>"
                        icone="<span class='glyphicons glyphicons-disk-saveSVG'>"
                        longtitle="sauvegarder le contenu de l'éditeur dans un fichier"
                        onclick="saveSVGDiagram('%(id)s');"><span class='glyphicons glyphicons-disk-save'></span></button>
                    <button class='btn btn-warning' id="%(id)s-file-btn-upload"  style='background-color : rgb(234,138,0);' 
                        title="sauvegarder le diagramme au format PNG"
                        command="<kbd>Ctrl+S</kbd>"
                        icone="<span class='glyphicons glyphicons-disk-save'>"
                        longtitle="sauvegarder le contenu de l'éditeur dans un fichier"
                        onclick="savePNGDiagram('%(id)s');"><span class='glyphicons glyphicons-screenshot'></span></button>
                    <button class='btn btn-success' id="%(id)s-file-btn-download" style='background-color : #5C885C;'  
                        title="analyser le diagramme"
                        command="<kbd>Ctrl+O</kbd>"
                        icone="<span class='glyphicons glyphicons-play'></span>"
                        longtitle="vérification"
                        onclick="runDiagram('%(id)s');"><span class='glyphicons glyphicons-play'></span></button>
                </span>
            </th>
        </thead>
        <tbody>
            <tr>
                <td valign="top">
                    <div id="%(id)s-diagram-overview" name="%(name)s-diagram-overview"
                        class="diagram-overview %(class)s-diagram-overview" 
                        style="height:150px; border:1px solid blue;">
                    </div>
                </td>
                <td valign="top">
                    <textarea id="%(id)s-diagram-code" name="%(name)s-diagram-code"
                        style="display:none;" placeholder="code JSON">%(code)s</textarea>
                    <div id="%(id)s-diagram-feedback" name="%(name)s-diagram-feedback"
                        class="diagram-feedback %(class)s-diagram-feedback"></div>
                </td>
            </tr>
        </tbody>
    </table>
"""

DIAGRAM_HTML_END = u"""
</div>
<script type="text/javascript">
    $(document).ready( function() { editor["%(id)s"] = initDiagram("%(id)s"); } );
</script>
"""

DIAGRAM_LATEX_BEGIN = u"""
"""

DIAGRAM_LATEX_END = u"""
"""

#------------------------------------------------------------------------------
class diagram(nodes.General, nodes.Element):
    def __init__(self,options):
        super(diagram,self).__init__()
        self.options = options

def visit_html_diagram(self, node):
    res = node.html_begin % node.options
    self.body.append(res)

def depart_html_diagram(self,node):
    res = node.html_end % node.options
    self.body.append(res)

def visit_latex_diagram(self, node):
    pass
def depart_latex_diagram(self,node):
    pass
#-----------------------------------------------------
def versiondiagram(argument) :
    return directives.choice(argument, ("full","light"))

def typediagram(argument) :
    return directives.choice(argument, 
        ("analogic","grafcet","logic","mindmap","petri","uml"))

class Diagram(EniBook):
    required_arguments = 1
    optional_arguments = 0
    final_argument_whitespace = True
    has_content = True
    option_spec = {
        'class'   : directives.class_option,
        'name'    : directives.unchanged,
        'file'    : directives.path,
        'version' : versiondiagram
    }

    def run(self):
        super(Diagram,self).run()
        
        if typediagram(self.arguments[0]) :
            self.options['type'] = self.arguments[0]

        if 'version' not in self.options :
            self.options['version'] = 'full'

        if 'file' not in self.options :
            self.options['file'] = '' 

        if self.options['file'] != '' :
            text = self.include(self.options['file'])
            self.options['code'] = text
        else :
            self.options['code'] = ''

        node = diagram(self.options)
        node.html_begin  = DIAGRAM_HTML_BEGIN
        node.html_end    = DIAGRAM_HTML_END
        node.latex_begin = DIAGRAM_LATEX_BEGIN
        node.latex_end   = DIAGRAM_LATEX_END
        self.state.nested_parse(self.content, self.content_offset, node)
        return [node]
#------------------------------------------------------------------------------

